using System;
using System.IO;
using System.Text;
using gov.va.med.vbecs.Common.Log;

namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Brian    lin</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>5/9/2006</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///The HL7AckMessage type represents the HL7 v2.4 General Acknowledgement message.
	///</summary>

	#endregion

	public sealed class HL7AckMessage : HL7ProtocolMessage
	{
        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		private const bool RESPONSE_REQUIRED_INDICATOR = true;
		private const string CARRIAGE_RETURN = "\x0D";

		private string _msh;
		private string _msa;


		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static HL7AckMessage() {}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3"> 
		///		<ExpectedInput>Non-null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>HL7AckMessage object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="15"> 
		///		<ExpectedInput>Null msg input string.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// HL7Acknowledgement Message
		/// </summary>
		public HL7AckMessage( string message )
			:base( message )
		{
			if( message == null )
				throw( new ArgumentNullException( "message" ) );
			//
			_message = message;
			//
			// Split the message into segments
			LoadDataFromHL7Message();
			//
			_messageControlID = HL7Utility.GetMessageControlID( message );
			//
			try
			{
				// This includes field separator in position [0] and encoding characters in [1] - [4]
				char [] delimiters = HL7Utility.ParseGetMessageDelimiters(message);
				//
				string [] MshArray = MSH.Split(delimiters[0]);
				string [] messageTypeAndTriggerEvent = MshArray[9].Split(delimiters[1]);
				//
				if ( messageTypeAndTriggerEvent != null && messageTypeAndTriggerEvent.Length > 0 )
				{
					_messageType = HL7Utility.ConvertString( messageTypeAndTriggerEvent[0] );
				}
			}
			catch(Exception ex)
			{
				var innerException = HL7Utility.GetInnerExceptionFromException(ex);
				// EventLogAppender: note that I added reference to VBECS.Common
                _eventsLogger.Error("Error processing HL7AckMessage.\n\n" + innerException);
				//System.Diagnostics.EventLog.WriteEntry( "Error processing HL7AckMessage.\n\n", innerException, System.Diagnostics.EventLogEntryType.Error );
			}
		}

		/// <summary>
		/// 
		/// </summary>
		private void LoadDataFromHL7Message()
		{
			_msh = HL7Utility.ParseGetRequiredMessageSegment(this.Message, SegmentTypeNames.MSH);
			_msa = HL7Utility.ParseGetRequiredMessageSegment(this.Message, SegmentTypeNames.MSA);
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="121"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Message string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="122"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overridden method to get the full hl7 message.
		/// </summary>
		/// <returns></returns>
		public override string GetMessage()
		{
			return this.Message;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="149"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="150"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overridden method indicating if the response is required
		/// </summary>
		/// <returns></returns>
		public override bool IsResponseRequired()
		{
			return RESPONSE_REQUIRED_INDICATOR;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="193"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="194"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overridden method to get the Message Control ID
		/// </summary>
		/// <returns></returns>
		public override string GetMessageControlID()
		{
			return MessageControlID;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="205"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="206"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overriden method to get the Message Type.
		/// </summary>
		/// <returns></returns>
		public override string GetMessageType()
		{
			return MessageType;
		}

		/// <summary>
		/// WriteHL7MessageBody
		/// </summary>
		/// <param name="writer"></param>
		protected override void WriteHL7MessageBody( StreamWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.Write( this.Message );
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="212"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="213"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the Message Type.
		/// </summary>
		public string MessageType
		{
			get
			{
				return _messageType;
			}

		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="250"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="251"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the Message.
		/// </summary>
		public string Message
		{
			get
			{
				return _message;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="255"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="278"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the MessageControlID.
		/// </summary>
		public string MessageControlID
		{
			get
			{
				return _messageControlID;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="279"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="287"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the MSH message segment
		/// </summary>
		public string MSH
		{
			get
			{
				return _msh;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="301"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="302"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the MSA message segment
		/// </summary>
		public string MSA
		{
			get
			{
				return _msa;
			}
		}

		/// <summary>
		/// This method supports CR 2140. The method creates a generic ACK message
		/// that is returned to the sending system when an HL7 message is received that
		/// includes either a sending or receiving application in the MSH segment that is
		/// not supported by VBECS. An Application Error message is returned.
		/// CR 2960 also.
		/// </summary>
		/// <param name="hl7Message"></param>
		/// <param name="ackCode"></param>
		/// <param name="errorMessage"></param>
		/// <returns></returns>
		public static HL7AckMessage BuildAckMessage( string hl7Message, AckCodes ackCode, string errorMessage )
		{
			if( hl7Message == null )
			{
				throw( new ArgumentNullException( "hl7Message" ) );
			}
			//
			char [] delimiters = HL7Utility.ParseGetMessageDelimiters( hl7Message );
			//
			string fieldSeparator = delimiters[0].ToString();
			//
			string encodingCharacters = string.Concat( delimiters[1].ToString(), delimiters[2].ToString(), delimiters[3].ToString(), delimiters[4].ToString() );
			//
			string mshSegment = HL7Utility.ParseGetRequiredMessageSegment( hl7Message, SegmentTypeNames.MSH );
			//
			// CR 3106: it's possible we can't determine Message Control Id if message was really messed up
			string messageControlId = null;
			try { messageControlId = HL7Utility.GetMessageControlID(hl7Message); }
			catch(Exception) {} // just ignore and use null
			//
			StringBuilder ack = new StringBuilder();
			ack.Append("MSH");
			ack.Append(fieldSeparator);
			ack.Append(encodingCharacters);
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 4));
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 5));
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 2));
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 3));
			ack.Append(fieldSeparator);
			ack.Append(System.DateTime.Now.ToString("yyyyMMddhhmmsszzz").Replace(":",""));		
			ack.Append(fieldSeparator);
			ack.Append(fieldSeparator);
			ack.Append("ACK");
			ack.Append(fieldSeparator);
			ack.Append(string.Concat("VBECS", System.DateTime.Now.ToString("yMMddhhssffff")));
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 10));
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 11));
			ack.Append(fieldSeparator);
			ack.Append(fieldSeparator);
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 14));
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 15));
			ack.Append(fieldSeparator);
			ack.Append(HL7Utility.ParseGetSegmentData(mshSegment, fieldSeparator, 16));
			ack.Append("\x0D");
			ack.Append("MSA");
			ack.Append(fieldSeparator);
			ack.Append(ackCode.ToString());
			ack.Append(fieldSeparator);
			ack.Append(messageControlId); // CR 3006
			ack.Append(fieldSeparator);
			ack.Append(errorMessage);
			ack.Append("\x0D");
			//
			return new HL7AckMessage(ack.ToString());
		}
	}
}
